import os
import shutil
from pathlib import Path

VAULT_DIR = Path.home() / ".flux-tui" / "vault"

def init_vault():
    """Ensure the vault directory exists."""
    if not VAULT_DIR.exists():
        VAULT_DIR.mkdir(parents=True, exist_ok=True)

def list_vault_files():
    """Return a list of files in the vault with their sizes."""
    init_vault()
    files = []
    for item in VAULT_DIR.iterdir():
        if item.is_file():
            size_kb = item.stat().st_size / 1024
            files.append({
                'name': item.name,
                'size': f"{size_kb:.2f} KB",
                'path': str(item)
            })
    return files

def get_vault_stats():
    """Return basic statistics about the vault."""
    files = list_vault_files()
    return {
        'count': len(files),
        'location': str(VAULT_DIR)
    }

def delete_vault_file(filename):
    """Delete a file from the vault."""
    file_path = VAULT_DIR / filename
    if file_path.exists():
        file_path.unlink()
        return True
    return False

def move_to_vault(file_path):
    """Move an encrypted file to the vault directory."""
    init_vault()
    dest_path = VAULT_DIR / Path(file_path).name
    if dest_path.exists():
        # Avoid overwriting existing files in the vault with same name
        count = 1
        while True:
            new_name = f"{Path(file_path).stem}_{count}{Path(file_path).suffix}"
            dest_path = VAULT_DIR / new_name
            if not dest_path.exists():
                break
            count += 1
            
    shutil.move(file_path, dest_path)
    return str(dest_path)
