from rich.layout import Layout
from rich.panel import Panel
from rich.table import Table
from rich.text import Text
from rich.console import Console, Group
from rich.align import Align
from rich.box import ROUNDED, DOUBLE_EDGE
from rich.theme import Theme
from datetime import datetime
import platform

# Elite UI Module for FLUX-TUI (v22 - Elite Security Edition)
# Designed with a unified purple/white theme for a premium security experience

class FluxUI:
    def __init__(self, console: Console = None):
        if console is None:
            # Unified Elite Theme: Removing all blue/cyan, using Purple/White/Green/Red
            custom_theme = Theme({
                "primary": "bold magenta",
                "secondary": "bold white",
                "success": "bold green",
                "error": "bold red",
                "warning": "bold yellow",
                "info": "bold magenta",
                "dim": "dim white"
            })
            self.console = Console(theme=custom_theme)
        else:
            self.console = console
            
        self.accent_color = "magenta"
        self.text_color = "white"

    def make_header(self) -> Panel:
        header_text = Text.assemble(
            (" FLUX", "bold magenta"),
            ("-", "white"),
            ("TUI ", "bold magenta"),
            "| ",
            ("SECURE OFFLINE VAULT ", "italic white")
        )
        sys_info = f"[dim]{platform.system()} | {platform.machine()} | {datetime.now().strftime('%H:%M:%S')}[/dim]"
        header_group = Group(Align.center(header_text), Align.center(sys_info))
        return Panel(header_group, style="magenta", box=ROUNDED, padding=(0, 1))

    def make_sidebar(self, stats: dict = None) -> Panel:
        if stats is None:
            stats = {"Files": "0", "Total Size": "0 KB", "Status": "Ready"}
        table = Table.grid(expand=True)
        table.add_column(style="bold white", width=12)
        table.add_column(style="white")
        for key, value in stats.items():
            table.add_row(f"{key}:", str(value))
            table.add_row("", "")
        tips = Text.assemble(
            ("\nQuick Tips:\n", "bold magenta"),
            ("• Drag & Drop files\n", "dim white"),
            ("• Type 'help' for list\n", "dim white"),
            ("• 'exit' to close\n", "dim white")
        )
        sidebar_group = Group(
            Panel(table, title="[bold magenta]VAULT STATS[/bold magenta]", border_style="magenta", box=ROUNDED),
            Panel(tips, title="[bold magenta]INFO[/bold magenta]", border_style="magenta", box=ROUNDED)
        )
        return Panel(sidebar_group, border_style="magenta", box=ROUNDED)

    def make_main_dashboard(self, content=None) -> Panel:
        if content is None:
            welcome_text = Text.assemble(
                ("\n\nWelcome to the FLUX Secure Environment\n", "bold magenta"),
                ("Authenticated AES-256-GCM Protection\n\n", "italic white"),
                ("Your data is encrypted locally with no external access.\n", "dim white"),
                ("Drag a file here or type a command below to begin.", "magenta")
            )
            content = Align.center(welcome_text, vertical="middle")
        return Panel(content, title="[bold magenta]SECURE TERMINAL[/bold magenta]", border_style="magenta", box=DOUBLE_EDGE)

    def make_input_area(self, last_status="Ready") -> Panel:
        """Creates an elite, purple-themed input area with no blue text."""
        input_text = Text.assemble(
            (" STATUS: ", "bold black on magenta"),
            (f" {last_status.upper()} ", "bold white on black"),
            (" | ", "dim white"),
            (" COMMAND: ", "bold magenta")
        )
        return Panel(input_text, title="[bold magenta]COMMAND INPUT[/bold magenta]", border_style="magenta", box=ROUNDED, padding=(0, 1))

    def get_layout(self, main_content=None, sidebar_stats=None, footer_msg="Ready") -> Layout:
        layout = Layout()
        layout.split(
            Layout(name="header", size=4),
            Layout(name="body"),
            Layout(name="input", size=3)
        )
        layout["body"].split_row(
            Layout(name="sidebar", ratio=1),
            Layout(name="main", ratio=3)
        )
        layout["header"].update(self.make_header())
        layout["sidebar"].update(self.make_sidebar(sidebar_stats))
        layout["main"].update(self.make_main_dashboard(main_content))
        layout["input"].update(self.make_input_area(footer_msg))
        return layout

    def display_auth_screen(self, is_setup=False):
        self.console.clear()
        title = "INITIAL SECURITY SETUP" if is_setup else "SECURE ACCESS REQUIRED"
        auth_text = Text.assemble(
            ("\n" + title + "\n", "bold magenta"),
            ("-------------------------\n", "magenta"),
            ("Enter your Master Security Key to unlock the vault.\n", "dim white")
        )
        panel = Panel(Align.center(auth_text), border_style="magenta", box=DOUBLE_EDGE, padding=(2, 4))
        self.console.print("\n" * 2)
        self.console.print(Align.center(panel))
        self.console.print("\n")

def get_console():
    return FluxUI().console

def print_message(message, type="info"):
    ui = FluxUI()
    icons = {"info": "ℹ", "success": "✔", "error": "✖", "warning": "⚠"}
    styles = {"info": "magenta", "success": "green", "error": "red", "warning": "yellow"}
    icon = icons.get(type, "•")
    style = styles.get(type, "white")
    ui.console.print(f"[{style}]{icon} {message}[/{style}]")
