import json
import os
from pathlib import Path

CONFIG_DIR = Path.home() / ".flux-tui"
SETTINGS_FILE = CONFIG_DIR / "settings.json"

DEFAULT_SETTINGS = {
    'theme': 'dark',
}

def init_settings():
    """Ensure settings file exists with defaults."""
    if not CONFIG_DIR.exists():
        CONFIG_DIR.mkdir(parents=True, exist_ok=True)
    
    if not SETTINGS_FILE.exists():
        save_settings(DEFAULT_SETTINGS)

def load_settings():
    """Load settings from the configuration file."""
    init_settings()
    try:
        if SETTINGS_FILE.exists():
            with open(SETTINGS_FILE, 'r') as f:
                return json.load(f)
        return DEFAULT_SETTINGS
    except Exception:
        return DEFAULT_SETTINGS

def save_settings(settings):
    """Save settings to the configuration file."""
    if not CONFIG_DIR.exists():
        CONFIG_DIR.mkdir(parents=True, exist_ok=True)
    with open(SETTINGS_FILE, 'w') as f:
        json.dump(settings, f, indent=4)

def toggle_theme():
    """Switch between dark and light themes."""
    settings = load_settings()
    settings['theme'] = 'light' if settings['theme'] == 'dark' else 'dark'
    save_settings(settings)
    return settings['theme']

def get_theme():
    """Return the current theme preference."""
    settings = load_settings()
    return settings.get('theme', 'dark')
