import os
from cryptography.hazmat.primitives.ciphers.aead import AESGCM
from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.backends import default_backend

# Professional Crypto Module for FLUX-TUI (v21 - In-Memory Decryption)
# High-grade AES-256-GCM protection with secure key derivation

SALT_SIZE = 16
NONCE_SIZE = 12

def derive_key(password: str, salt: bytes) -> bytes:
    """Derive a 256-bit key from the password and salt."""
    kdf = PBKDF2HMAC(
        algorithm=hashes.SHA256(),
        length=32,
        salt=salt,
        iterations=100000,
        backend=default_backend()
    )
    return kdf.derive(password.encode())

def encrypt_file(file_path: str, password: str, output_path: str):
    """Encrypt a file using AES-256-GCM."""
    salt = os.urandom(SALT_SIZE)
    nonce = os.urandom(NONCE_SIZE)
    key = derive_key(password, salt)
    aesgcm = AESGCM(key)
    
    with open(file_path, 'rb') as f:
        data = f.read()
    
    ciphertext = aesgcm.encrypt(nonce, data, None)
    
    with open(output_path, 'wb') as f:
        f.write(salt + nonce + ciphertext)

def decrypt_file(file_path: str, password: str, output_path: str):
    """Decrypt a file using AES-256-GCM."""
    with open(file_path, 'rb') as f:
        content = f.read()
    
    salt = content[:SALT_SIZE]
    nonce = content[SALT_SIZE:SALT_SIZE + NONCE_SIZE]
    ciphertext = content[SALT_SIZE + NONCE_SIZE:]
    
    key = derive_key(password, salt)
    aesgcm = AESGCM(key)
    
    decrypted_data = aesgcm.decrypt(nonce, ciphertext, None)
    
    with open(output_path, 'wb') as f:
        f.write(decrypted_data)

def decrypt_to_memory(file_path: str, password: str) -> str:
    """Decrypts a file in-memory and returns the content as a string (for 'view' command)."""
    with open(file_path, 'rb') as f:
        content = f.read()
    
    salt = content[:SALT_SIZE]
    nonce = content[SALT_SIZE:SALT_SIZE + NONCE_SIZE]
    ciphertext = content[SALT_SIZE + NONCE_SIZE:]
    
    key = derive_key(password, salt)
    aesgcm = AESGCM(key)
    
    try:
        decrypted_data = aesgcm.decrypt(nonce, ciphertext, None)
        # Attempt to decode as UTF-8, replacing errors for binary files
        return decrypted_data.decode('utf-8', errors='replace')
    except Exception:
        raise Exception("Incorrect Security Key or Corrupted File")
