import os
import time
import sys
import platform
from pathlib import Path
from rich.console import Console, Group
from rich.panel import Panel
from rich.table import Table
from rich.text import Text
from rich.align import Align

from .ui import FluxUI, print_message
from .crypto import encrypt_file, decrypt_file, decrypt_to_memory
from .vault import get_vault_stats, list_vault_files, move_to_vault, delete_vault_file, VAULT_DIR
from .settings import toggle_theme
from .auth import save_master_password, verify_master_password

def clean_path(path_str):
    return path_str.strip().strip('"').strip("'")

def set_terminal_title(title: str):
    if platform.system() == "Windows":
        os.system(f"title {title}")
    else:
        sys.stdout.write(f"\x1b]2;{title}\x07")
        sys.stdout.flush()

class FluxCLI:
    def __init__(self, master_password):
        self.master_password = master_password
        self.ui = FluxUI()
        self.console = self.ui.console
        self.status_msg = "READY"
        self.current_content = None
        set_terminal_title("FLUX-TUI | Elite Security Vault")

    def display_dashboard(self):
        """Renders the single-screen elite dashboard."""
        stats = get_vault_stats()
        ui_stats = {
            "Files": str(stats.get('count', 0)),
            "Total Size": stats.get('size', '0 KB'),
            "Status": "SECURE"
        }
        layout = self.ui.get_layout(
            main_content=self.current_content,
            sidebar_stats=ui_stats,
            footer_msg=self.status_msg
        )
        self.console.clear()
        self.console.print(layout)

    def move_cursor_to_input(self):
        """Moves the cursor into the purple input box with precise alignment."""
        # Standard ANSI escape to move cursor: \033[Line;ColumnH
        # However, relative movement is safer across different terminal sizes.
        # Move up 2 lines from the bottom, then move right 35 chars.
        sys.stdout.write("\033[2A\033[35C")
        sys.stdout.flush()

    def run(self):
        """Elite command loop with in-box typing and fixed command processing."""
        while True:
            try:
                self.display_dashboard()
                self.move_cursor_to_input()
                
                cmd_input = input().strip()
                if not cmd_input:
                    continue
                
                parts = cmd_input.split(maxsplit=1)
                cmd = parts[0].lower()
                args = parts[1] if len(parts) > 1 else ""

                if cmd in ["exit", "quit"]:
                    self.status_msg = "SECURELY CLOSING..."
                    self.display_dashboard()
                    time.sleep(0.8)
                    break
                
                elif cmd in ["help", "?", "man"]:
                    self.handle_help()
                    
                elif cmd == "encrypt":
                    self.handle_encrypt(args)
                    
                elif cmd == "decrypt":
                    self.handle_decrypt(args)
                    
                elif cmd == "view":
                    self.handle_view(args)
                    
                elif cmd in ["vault", "list"]:
                    self.handle_list_vault()
                    
                elif cmd == "delete":
                    self.handle_delete(args)
                    
                elif cmd == "export":
                    self.handle_export(args)
                    
                elif cmd == "theme":
                    toggle_theme()
                    self.status_msg = "THEME UPDATED"
                    
                elif cmd == "changepassword":
                    self.handle_change_password()
                    
                elif cmd in ["dashboard", "clear"]:
                    self.current_content = None
                    self.status_msg = "DASHBOARD REFRESHED"
                    
                else:
                    self.status_msg = f"UNKNOWN COMMAND: '{cmd.upper()}'"
                
            except KeyboardInterrupt:
                self.status_msg = "USE 'EXIT' TO CLOSE"
            except Exception as e:
                self.status_msg = f"ERROR: {str(e).upper()}"

    def handle_help(self):
        """Elite Help System: Detailed command usage and descriptions."""
        table = Table(title="[bold magenta]ELITE COMMAND REFERENCE[/bold magenta]", border_style="magenta", expand=True, box=None)
        table.add_column("COMMAND", style="bold magenta", width=18)
        table.add_column("USAGE", style="bold white", width=25)
        table.add_column("DESCRIPTION", style="dim white")
        
        table.add_row("encrypt", "encrypt [path]", "Encrypt a file and secure it in the vault")
        table.add_row("decrypt", "decrypt [name]", "Restore a file from vault to current folder")
        table.add_row("view", "view [name]", "Read an encrypted text/note in-terminal")
        table.add_row("vault", "vault", "List all secured files and their sizes")
        table.add_row("export", "export [name] [path]", "Decrypt and save to a specific location")
        table.add_row("delete", "delete [name]", "Permanently remove a file from the vault")
        table.add_row("theme", "theme", "Switch between high-contrast UI themes")
        table.add_row("changepassword", "changepassword", "Update your Master Security Key")
        table.add_row("exit", "exit", "Securely wipe session and close app")
        
        footer = Text("\nTip: You can drag and drop files directly into the box for encrypt/decrypt.", style="italic magenta")
        # Ensure we update the current content and status correctly
        self.current_content = Group(table, Align.center(footer))
        self.status_msg = "HELP SYSTEM ACTIVE"

    def handle_encrypt(self, args):
        if not args:
            self.status_msg = "DRAG FILE INTO BOX..."
            self.display_dashboard()
            self.move_cursor_to_input()
            path = input().strip()
        else:
            path = args
            
        path = clean_path(path)
        if not os.path.exists(path):
            self.status_msg = "ERROR: FILE NOT FOUND"
            return
            
        try:
            filename = Path(path).name
            self.status_msg = f"ENCRYPTING {filename.upper()}..."
            self.display_dashboard()
            
            temp_encrypted = f"{filename}.flux"
            encrypt_file(path, self.master_password, temp_encrypted)
            
            self.status_msg = "SECURING TO VAULT..."
            self.display_dashboard()
            final_path = move_to_vault(temp_encrypted)
            
            self.status_msg = "ENCRYPTION SUCCESSFUL"
            self.current_content = Align.center(
                Text.assemble(
                    ("ELITE PROTECTION ACTIVE\n\n", "bold green"),
                    (f"File: {Path(final_path).name}\n", "white"),
                    (f"Method: AES-256-GCM\n\n", "dim white"),
                    ("The encrypted file is now safely stored in your vault.", "dim")
                ), vertical="middle"
            )
        except Exception as e:
            self.status_msg = "ENCRYPTION FAILED"
            self.current_content = Align.center(Text(f"Critical Error: {str(e)}", style="bold red"), vertical="middle")

    def handle_decrypt(self, args):
        if not args:
            self.status_msg = "ENTER FILENAME..."
            self.display_dashboard()
            self.move_cursor_to_input()
            path = input().strip()
        else:
            path = args
            
        path = clean_path(path)
        if not os.path.exists(path):
            vault_path = VAULT_DIR / path
            if vault_path.exists():
                path = str(vault_path)
            else:
                self.status_msg = "ERROR: NOT IN VAULT"
                return
            
        try:
            filename = Path(path).name
            self.status_msg = f"DECRYPTING {filename.upper()}..."
            self.display_dashboard()
            
            output_name = filename.replace(".flux", "")
            if output_name == filename:
                output_name = f"decrypted_{output_name}"
            
            decrypt_file(path, self.master_password, output_name)
            self.status_msg = "DECRYPTION SUCCESSFUL"
            self.current_content = Align.center(
                Text.assemble(
                    ("FILE RESTORED SUCCESSFULLY\n\n", "bold green"),
                    (f"Target: {output_name}\n\n", "white"),
                    ("The file has been decrypted and returned to your local directory.", "dim")
                ), vertical="middle"
            )
        except Exception as e:
            self.status_msg = "DECRYPTION FAILED"
            self.current_content = Align.center(Text("Error: Incorrect Master Key or Corrupted Data", style="bold red"), vertical="middle")

    def handle_view(self, args):
        if not args:
            self.status_msg = "ENTER FILENAME TO VIEW..."
            self.display_dashboard()
            self.move_cursor_to_input()
            filename = input().strip()
        else:
            filename = args
            
        vault_path = VAULT_DIR / filename
        if not vault_path.exists():
            if not filename.endswith(".flux"):
                vault_path = VAULT_DIR / f"{filename}.flux"
            if not vault_path.exists():
                self.status_msg = "ERROR: NOT IN VAULT"
                return
            
        try:
            self.status_msg = f"DECRYPTING FOR VIEW..."
            self.display_dashboard()
            
            content = decrypt_to_memory(str(vault_path), self.master_password)
            view_panel = Panel(
                Text(content, style="white"),
                title=f"[bold magenta]SECURE VIEW: {vault_path.name}[/bold magenta]",
                border_style="magenta",
                padding=(1, 2)
            )
            self.current_content = view_panel
            self.status_msg = "SECURE VIEW ACTIVE"
        except Exception as e:
            self.status_msg = "VIEW FAILED"
            self.current_content = Align.center(Text(f"Error: {str(e)}", style="bold red"), vertical="middle")

    def handle_list_vault(self):
        files = list_vault_files()
        if not files:
            self.current_content = Align.center(Text("The vault is currently empty.", style="warning"), vertical="middle")
            self.status_msg = "VAULT EMPTY"
            return

        table = Table(title="[bold magenta]SECURE VAULT INVENTORY[/bold magenta]", border_style="magenta", expand=True)
        table.add_column("ID", style="dim", width=4)
        table.add_column("File Name", style="white")
        table.add_column("Size", justify="right", style="magenta")
        
        for idx, f in enumerate(files, 1):
            table.add_row(str(idx), f['name'], f['size'])
            
        self.current_content = table
        self.status_msg = f"VAULT INVENTORY: {len(files)} FILES"

    def handle_delete(self, args):
        if not args:
            self.status_msg = "ENTER FILENAME..."
            self.display_dashboard()
            self.move_cursor_to_input()
            filename = input().strip()
        else:
            filename = args
            
        if delete_vault_file(filename):
            self.status_msg = "PERMANENTLY REMOVED"
            self.current_content = Align.center(Text(f"File '{filename}' has been purged from vault.", style="bold green"), vertical="middle")
        else:
            self.status_msg = "ERROR: NOT FOUND"

    def handle_export(self, args):
        parts = args.split(maxsplit=1)
        if len(parts) < 2:
            self.status_msg = "USAGE: export [name] [dest]"
            self.display_dashboard()
            return
            
        filename, dest = parts[0], parts[1]
        vault_path = VAULT_DIR / filename
        if not vault_path.exists():
            if not filename.endswith(".flux"):
                vault_path = VAULT_DIR / f"{filename}.flux"
            if not vault_path.exists():
                self.status_msg = "ERROR: NOT IN VAULT"
                return
            
        dest = clean_path(dest)
        try:
            self.status_msg = f"EXPORTING TO {dest.upper()}..."
            self.display_dashboard()
            
            output_name = filename.replace(".flux", "")
            dest_path = os.path.join(dest, output_name) if os.path.isdir(dest) else dest
            
            decrypt_file(str(vault_path), self.master_password, dest_path)
            self.status_msg = "EXPORT SUCCESSFUL"
            self.current_content = Align.center(Text(f"File exported to: {dest_path}", style="bold green"), vertical="middle")
        except Exception as e:
            self.status_msg = "EXPORT FAILED"

    def handle_change_password(self):
        self.status_msg = "VERIFY CURRENT KEY..."
        self.display_dashboard()
        self.move_cursor_to_input()
        old_pwd = input().strip()
        if not verify_master_password(old_pwd):
            self.status_msg = "ACCESS DENIED"
            return
            
        self.status_msg = "ENTER NEW KEY..."
        self.display_dashboard()
        self.move_cursor_to_input()
        new_pwd = input().strip()
        self.status_msg = "CONFIRM NEW KEY..."
        self.display_dashboard()
        self.move_cursor_to_input()
        confirm_pwd = input().strip()
        
        if new_pwd != confirm_pwd:
            self.status_msg = "KEYS DO NOT MATCH"
            return
            
        save_master_password(new_pwd)
        self.master_password = new_pwd
        self.status_msg = "KEY UPDATED"
        self.current_content = Align.center(Text("Master Security Key updated successfully.", style="bold green"), vertical="middle")
