import hashlib
import os
import json
from pathlib import Path

CONFIG_DIR = Path.home() / ".flux-tui"
AUTH_FILE = CONFIG_DIR / "auth.json"

def init_auth_system():
    if not CONFIG_DIR.exists():
        CONFIG_DIR.mkdir(parents=True, exist_ok=True)

def hash_password(password, salt=None):
    if salt is None:
        salt = os.urandom(32)
    pwd_hash = hashlib.pbkdf2_hmac(
        'sha256',
        password.encode('utf-8'),
        salt,
        100000
    )
    return salt, pwd_hash

def save_master_password(password):
    init_auth_system()
    salt, pwd_hash = hash_password(password)
    with open(AUTH_FILE, 'w') as f:
        json.dump({
            'salt': salt.hex(),
            'hash': pwd_hash.hex()
        }, f)

def verify_master_password(password):
    if not AUTH_FILE.exists():
        return False
    try:
        with open(AUTH_FILE, 'r') as f:
            data = json.load(f)
        
        salt = bytes.fromhex(data['salt'])
        stored_hash = bytes.fromhex(data['hash'])
        
        _, current_hash = hash_password(password, salt)
        return current_hash == stored_hash
    except Exception:
        return False

def is_first_run():
    return not AUTH_FILE.exists()
