import sys
import os
import time
from src.auth import is_first_run, save_master_password, verify_master_password
from src.ui import FluxUI, print_message
from src.cli import FluxCLI, set_terminal_title
from src.settings import init_settings

def main():
    try:
        # Initialize professional settings and title
        init_settings()
        set_terminal_title("FLUX-TUI | Secure Access")
        
        ui = FluxUI()
        
        if is_first_run():
            ui.display_auth_screen(is_setup=True)
            while True:
                # Use standard input for 100% reliability
                print("CREATE NEW MASTER KEY ❯ ", end="", flush=True)
                pwd = input().strip()
                print("CONFIRM MASTER KEY ❯ ", end="", flush=True)
                confirm = input().strip()
                
                if not pwd:
                    print_message("Security key cannot be empty.", "error")
                    continue
                    
                if pwd == confirm:
                    save_master_password(pwd)
                    print_message("Master security key established successfully!", "success")
                    time.sleep(1)
                    master_password = pwd
                    break
                else:
                    print_message("Security keys do not match. Try again.", "error")
        else:
            attempts = 0
            while attempts < 3:
                ui.display_auth_screen(is_setup=False)
                if attempts > 0:
                    print_message(f"Access Denied: Incorrect key. {3 - attempts} attempts remaining.", "error")
                
                # Use standard input for 100% reliability
                print("ENTER MASTER KEY ❯ ", end="", flush=True)
                master_password = input().strip()
                
                if verify_master_password(master_password):
                    print_message("Vault access granted!", "success")
                    time.sleep(0.5)
                    break
                else:
                    attempts += 1
            else:
                print_message("CRITICAL: Too many failed attempts. Locking system.", "error")
                time.sleep(2)
                sys.exit(1)

        # Start the professional CLI
        cli = FluxCLI(master_password)
        cli.run()
        
    except KeyboardInterrupt:
        print("\n\nFLUX-TUI Securely exiting...")
        sys.exit(0)
    except Exception as e:
        print(f"\n[FATAL ERROR] System failure: {str(e)}")
        # Keep the error visible for the user
        input("\nPress Enter to exit...")
        sys.exit(1)

if __name__ == "__main__":
    main()
