import os
import sys
import subprocess
import platform
import shutil
from pathlib import Path

# Professional Build System for FLUX-TUI (v23 - Elite Stable Edition)
# Optimized for high-reliability native Windows builds with Python 3.13

def run_command(command, display_output=True):
    """Run a shell command with optional output display."""
    try:
        if display_output:
            subprocess.run(command, check=True, shell=True)
        else:
            subprocess.run(command, check=True, shell=True, capture_output=True)
    except subprocess.CalledProcessError as e:
        print(f"\n[CRITICAL ERROR] Failed to execute command: {e}")
        sys.exit(1)

def build():
    print("="*60)
    print("   FLUX-TUI: PROFESSIONAL WINDOWS NATIVE BUILD SYSTEM (v23)   ")
    print("="*60)
    
    # 1. Detect System and Python Version
    current_os = platform.system().lower()
    python_version = platform.python_version()
    print(f"\n[1/5] Detecting System: {current_os.upper()}")
    print(f"      Python Version: {python_version}")
    
    if current_os != "windows":
        print("\n[WARNING] This version of the build script is optimized for Windows.")

    # 2. Ensure dependencies are met
    print("\n[2/5] Initializing build environment...")
    print("Upgrading pip and installing PyInstaller, Rich, and Cryptography...")
    python_exe = f'"{sys.executable}"'
    run_command(f"{python_exe} -m pip install --quiet --upgrade pip", display_output=False)
    run_command(f"{python_exe} -m pip install --quiet pyinstaller rich cryptography", display_output=False)
    
    # 3. Build Configuration
    main_script = "main.py"
    icon_file = "ico.ico"
    app_name = "flux-tui"
    
    src_dir = Path(__file__).parent.absolute()
    os.chdir(src_dir)
    print(f"Working Directory: {src_dir}")

    # 4. Execute Build Process
    print(f"\n[3/5] Building {current_os.upper()} standalone executable...")
    
    separator = ";" if current_os == "windows" else ":"
    build_cmd = (
        f"{python_exe} -m PyInstaller --onefile --clean --noconfirm "
        f"--name {app_name} "
        f"--icon {icon_file} "
        f"--add-data \"src{separator}src\" "
        f"--distpath ./dist "
        f"{main_script}"
    )
    
    run_command(build_cmd)
    
    # 5. Post-Build Organization and Inno Setup
    print("\n[4/5] Organizing final artifacts and Installer...")
    
    dist_dir = src_dir / "dist"
    build_dir = src_dir / "build"
    spec_file = src_dir / f"{app_name}.spec"
    
    if dist_dir.exists():
        final_name = app_name
        if current_os == "windows":
            final_name = f"{app_name}.exe"
            target_name = f"flux-tui-windows.exe"
        else:
            target_name = f"flux-tui-{current_os}"
            
        if (dist_dir / final_name).exists():
            shutil.copy(dist_dir / final_name, dist_dir / target_name)
            print(f"Success! Standalone artifact created: dist/{target_name}")
            
            # Check for Inno Setup (ISCC.exe)
            if current_os == "windows":
                print("\n[INFO] Checking for Inno Setup (ISCC.exe) to build the installer...")
                iscc_path = shutil.which("ISCC.exe")
                iscc_locations = [
                    iscc_path,
                    "C:\\Program Files (x86)\\Inno Setup 6\\ISCC.exe",
                    "C:\\Program Files\\Inno Setup 6\\ISCC.exe"
                ]
                
                iscc = next((loc for loc in iscc_locations if loc and os.path.exists(loc)), None)
                
                if iscc:
                    print(f"Found Inno Setup at: {iscc}")
                    print("Building Professional Windows Installer...")
                    run_command(f'"{iscc}" setup.iss')
                    print("Success! Installer created in dist/installer/")
                else:
                    print("\n[NOTICE] Inno Setup not found. To build the professional installer:")
                    print("1. Download Inno Setup: https://jrsoftware.org/isdl.php")
                    print("2. Install it and run the included 'make_installer.bat'")
    
    # 6. Finalizing and cleaning workspace
    print("\n[5/5] Finalizing and cleaning workspace...")
    if build_dir.exists():
        shutil.rmtree(build_dir)
    if spec_file.exists():
        spec_file.unlink()

    print("="*60)
    print(f"   BUILD COMPLETE FOR {current_os.upper()}   ")
    print("="*60)

if __name__ == "__main__":
    build()
